;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  grid-x-inc  ;; the amount of patches in between two roads in the x direction
  grid-y-inc  ;; the amount of patches in between two roads in the y direction
  acceleration  ;; the constant that controls how much a car speeds up or slows down by if it is to accelerate or decelerate
  phase  ;; keeps track of the phase
  num-cars-stopped  ;; the number of cars that are stopped during a single pass thru the go procedure
  old-display-which-metric  ;; holds the value of display-which-metric for the last time through the go procedure

  ;; patch agentsets
  intersections  ;; agentset containing the patches that are intersections
  roads  ;; agentset containing the patches that are roads

  ;; string and list variables that hold data passed accumulated in the model run
  wait-data  ;; list that holds the average wait time of the cars for each pass through the go procedure
  stopped-data  ;; list that holds the number of stopped of the cars for each pass through the go procedure
  speed-data  ;; list that holds the average speed of the cars for each pass through the go procedure
  time-data  ;; list that holds the value of clock for each pass through the go procedure

  ;; variables for plotting client-defined metrics in the client plot
  current-client-metric  ;; the current metric that is being plotted in the client plot
  current-intersection-for-client-metric  ;; the intersection whose metric we are currently plotting in the client plot
  old-intersection-for-client-metric  ;; the last intersection for current-intersection-for-metric

  ;;quick start instructions variables
  quick-start  ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item  ;; the index of the current quickstart instruction
  qs-items  ;; the list of quickstart instructions
]

turtles-own
[
  speed  ;; the speed of the turtle
  up-car?  ;; this will be true if the turtle moves downwards and false if it moves to the right
  wait-time  ;; the amount of time since the last time a turtle has moved
]

patches-own
[
  intersection?  ;; this is true if the patch is at the intersection of two roads
  accident?  ;; this is true if a crash has occurred at this intersection.  this will never be true for a non-intersection patch
  green-light-up?  ;; this is true if the green light is above the intersection.  otherwise, it is false.  this is only true for patches that are intersections.
  my-row  ;; this holds the row of the intersection counting from the upper left corner of the View.  it is -1 for patches that are not intersections.
  my-column  ;; this holds the column of the intersection counting from the upper left corner of the View.  it is -1 for patches that are not intersections.
  user-id  ;; this holds the user-id that corresponds to the intersection.  it is -1 for patches that are not intersections.
  my-phase  ;; this holds the phase for the intersection.  it is -1 for patches that are not intersections.

  ;; client plot variables
  ;; this is a list of the values generated by the my-plot-metric that
  ;; will be plotted when this intersection's metric is plotted in the client plot
  my-plot-data
  ;; this is NetLogo code generates values for this intersection's values for my-plot-data.
  ;; the code should be a meaningful metric of how to measure traffic efficiency.
  my-plot-metric
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup
  setup-quick-start
  hubnet-reset
end

;; Initialize the display by giving the global and patch variables initial values.
;; Create num-cars of turtles if there are enough road patches for one turtle to be created per road patch.
;; Setup the plots
;; All code in setup is done if full-setup? is true.  If it is false, then it doesn't clear the information
;; about the users; users still retain the same light that they had before.
;; "setup false" is done by the re-run button.
to setup
  clear-output
  clear-turtles
  clear-all-plots

  let full-setup? ((grid-x-inc != (world-width / grid-size-x))
      or (grid-y-inc != (world-height / grid-size-y)))

  setup-globals

  ;; we only re-layout the grid if the sizes changed
  ifelse full-setup?
  [
    ;; if we have clients logged in we want to make sure to
    ;; keep the appropriate info around
    let users map [ p -> [user-id] of p] sort patches with [is-string? user-id]
    let plot-metrics map [ p -> [my-plot-metric] of p] sort patches with [is-string? user-id]
    let phases map [ p -> [my-phase] of p] sort patches with [is-string? user-id]
    clear-patches
    setup-patches
    setup-intersections
    ;; reassign the clients to intersections
    (foreach users plot-metrics phases [ [the-user the-plot-metric the-phase] ->
      get-free-intersection the-user
      ask intersections with [ user-id = the-user ]
      [
        set my-plot-metric the-plot-metric
        set my-phase the-phase
      ]
    ])
  ]
  [ setup-intersections ]

  set-default-shape turtles "car"

  if (number > count roads)
  [
    user-message (word "There are too many cars for the amount of road.  "
    "Either increase the amount of roads by increasing the GRID-SIZE-X "
    "or GRID-SIZE-Y sliders, or decrease the number of cars by lowering "
    "the NUMBER slider.\nThe setup has stopped.")
    stop
  ]

  ;; Now create the turtles and have each created turtle call the functions setup-cars and set-car-color
  create-turtles number
  [
    setup-cars
    set-car-color
    record-data
  ]

  ;; give the turtles an initial speed
  ask turtles
  [ set-car-speed ]

  update-list-info

  my-setup-plots
end

;; Initialize the global variables to appropriate values
to setup-globals
  reset-ticks
  set phase 0
  set num-cars-stopped 0
  set grid-x-inc world-width / grid-size-x
  set grid-y-inc world-height / grid-size-y

  ;; initialize the lists and string for HubNet
  set wait-data []
  set stopped-data []
  set speed-data []
  set time-data []

  set current-client-metric ""
  set current-intersection-for-client-metric nobody
  set old-intersection-for-client-metric nobody

  ;; don't make acceleration 0.1 since we could get a rounding error and end up on a patch boundary
  set acceleration 0.099
end

;; Make the patches have appropriate colors, setup the roads and intersections agentsets,
;; and initialize the traffic lights to one setting
to setup-patches
  ;; initialize the patch-own variables and color the patches to a base-color
  ask patches
  [
    set intersection? false
    set accident? false
    set green-light-up? true
    set my-row -1
    set my-column -1
    set user-id -1
    set my-phase -1
    set my-plot-data []
    set my-plot-metric ""

    set pcolor brown + 3
  ]

  ;; initialize the global variables that hold patch agentsets
  set roads patches with [ (floor ((pxcor + max-pxcor - floor(grid-x-inc - 1)) mod grid-x-inc) = 0) or
                           (floor ((pycor + max-pycor) mod grid-y-inc) = 0) ]
  set intersections roads with [ (floor ((pxcor + max-pxcor - floor(grid-x-inc - 1)) mod grid-x-inc) = 0) and
                                 (floor ((pycor + max-pycor) mod grid-y-inc) = 0) ]

  ask roads
  [ set pcolor white ]
end

;; Give the intersections appropriate values for the intersection?, my-row, and my-column
;; patch variables.  Make all the traffic lights start off so that the lights are red
;; horizontally and green vertically.
to setup-intersections
  ask intersections
  [
    set intersection? true
    set green-light-up? true
    set my-plot-data []
    set my-row floor ((pycor + max-pycor) / grid-y-inc )
    set my-column floor ((pxcor + max-pxcor) / grid-x-inc )
    set-signal-colors
  ]
end

;; Initialize the turtle variables to appropriate values and place the turtle on an empty road patch.
to setup-cars  ;; turtle procedure
  set speed 0
  set wait-time 0

  put-on-empty-road

  ifelse intersection?
  [
    ifelse random 2 = 1
    [ set up-car? true ]
    [ set up-car? false ]
  ]
  [
    ifelse (floor ((pxcor + max-pxcor - floor(grid-x-inc - 1)) mod grid-x-inc) = 0)
    [ set up-car? true ]
    [ set up-car? false ]
  ]

  ifelse up-car?
  [ set heading 180 ]
  [ set heading 90 ]
end

;; Find a road patch without any turtles on it and place the turtle there.
to put-on-empty-road  ;; turtle procedure
  move-to one-of roads
  if any? other turtles-here
  [ put-on-empty-road ]
end

;; Initialize the plots
to my-setup-plots
  set-current-plot "Stopped Cars"
  set-plot-y-range 0 number
  plot num-cars-stopped

  set-current-plot "Average Wait Time of Cars"
  plot mean [wait-time] of turtles

  set-current-plot "Average Speed of Cars"
  set-plot-y-range 0 speed-limit
  plot mean [speed] of turtles
end

;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; receives information from the clients and runs the simulation
to go
  every delay
  [
    ;; get commands and data from the clients
    listen-clients

    ;; clear any accidents from the last time thru the go procedure
    clear-accidents

    ;; if there are any intersections that are to switch automatically, have them change their color
    set-signals
    set num-cars-stopped 0

    ;; set the turtles speed for this time thru the procedure, move them forward their speed,
    ;; record data for plotting, and set the color of the turtles
    ;; to an appropriate color based on their speed
    ask turtles
    [
      set-car-speed
      fd speed
      record-data
      set-car-color
    ]
    ;; crash the cars if crash? is true
    if crash?
    [ crash-cars ]

    ;; update the information in the lists of plot data
    update-list-info

    ;; update the plots with the new information from this pass thru the procedure
    do-plotting
    ;; update the clock and the phase
    clock-tick
  ]
end

;; reports the amount of seconds by which to slow the model down
to-report delay
  ifelse simulation-speed <= 0
  [ report ln (10 / 0.001) ]
  [ report ln (10 / simulation-speed) ]
end

;; have the traffic lights change color if phase equals each intersections' my-phase
to set-signals
  ask intersections with
      [phase = floor ((my-phase * ticks-per-cycle) / 100) and (auto? or user-id = -1)]
  [
    set green-light-up? (not green-light-up?)
    set-signal-colors
  ]
end

;; This procedure checks the variable green-light-up? at each intersection and sets the
;; traffic lights to have the green light up or the green light to the left.
to set-signal-colors  ;; intersection (patch) procedure
  ifelse power?
  [
    ifelse green-light-up?
    [
      ask patch-at -1 0 [ set pcolor red ]
      ask patch-at 0 1 [ set pcolor green ]
    ]
    [
      ask patch-at -1 0 [ set pcolor green ]
      ask patch-at 0 1 [ set pcolor red ]
    ]
  ]
  [
    ask patch-at -1 0 [ set pcolor white ]
    ask patch-at 0 1 [ set pcolor white ]
  ]
end

;; set any intersection's color that had an accident back to white and make accident? false
to clear-accidents
  if crash?
  [
    ask patches with [accident?]
    [
      set pcolor white
      set accident? false
    ]
  ]
end

;; set the turtles' speed based on whether they are at a red traffic light or the speed of the
;; turtle (if any) on the patch in front of them
to set-car-speed  ;; turtle procedure
  ifelse pcolor = red
  [ set speed 0 ]
  [
    ifelse up-car?
    [ set-speed 0 -1 ]
    [ set-speed 1 0 ]
  ]
end

;; set the speed variable of the turtle to an appropriate value (not exceeding the
;; speed limit) based on whether there are turtles on the patch in front of the turtle
to set-speed [delta-x delta-y]  ;; turtle procedure

  let turtles-ahead turtles-on patch-at delta-x delta-y

  ;; if there are turtles in front of the turtle, slow down
  ;; otherwise, speed up
  ifelse any? turtles-ahead
  [
    let up-cars?-ahead [up-car?] of turtles-ahead
    ifelse member? up-car? up-cars?-ahead and member? (not up-car?) up-cars?-ahead
    [
      if not crash?
      [ set speed 0 ]
    ]
    [
      set speed [speed] of one-of turtles-ahead
      slow-down
    ]
  ]
  [ speed-up ]
end

;; decrease the speed of the turtle
to slow-down  ;; turtle procedure
  ifelse speed <= 0  ;;if speed < 0
  [ set speed 0 ]
  [ set speed speed - acceleration ]
end

;; increase the speed of the turtle
to speed-up  ;; turtle procedure
  ifelse speed > speed-limit
  [ set speed speed-limit ]
  [ set speed speed + acceleration ]
end

;; set the color of the turtle to a different color based on how fast the turtle is moving
to set-car-color  ;; turtle procedure
  ifelse speed < (speed-limit / 2)
  [ set color blue ]
  [ set color cyan - 2 ]
end

;; keep track of the number of stopped turtles and the amount of time a turtle has been stopped
;; if its speed is 0
to record-data  ;; turtle procedure
  ifelse speed = 0
  [
    set num-cars-stopped num-cars-stopped + 1
    set wait-time wait-time + 1
  ]
  [ set wait-time 0 ]
end

;; crash any turtles at the same intersection going in different directions
to crash-cars
  ask intersections with [any? turtles-here with [up-car?] and any? turtles-here with [not up-car?]]
  [
    set accident? true
    set pcolor orange
  ]
end

;; add the new information from this pass thru the go procedure to the HubNet lists
to update-list-info
  set wait-data lput (mean [wait-time] of turtles) wait-data
  set stopped-data lput num-cars-stopped stopped-data
  set speed-data lput (mean [speed] of turtles) speed-data
  set time-data lput ticks time-data
  ask intersections with [ my-plot-metric != "" ]
  [ add-new-client-metric-value ]
end

;; verify that my-plot-metric is a valid metric that will result in a number
;; and add that number to the my-plot-data list for the intersection
to add-new-client-metric-value  ;; intersection procedure
  let err __check-syntax (word "plotxy clock " my-plot-metric)
  ifelse err = ""
  [
    let new-value run-result my-plot-metric
    ifelse is-number? new-value
    [
      set my-plot-data lput list ticks new-value my-plot-data
      hubnet-send user-id "Current Metric Value" new-value
    ]
    [
      hubnet-send user-id "metric-error-message" (word my-plot-metric
        " does not produce a numeric value.")
      clear-client-plot-data
    ]
  ]
  [
    hubnet-send user-id "metric-error-message" (word my-plot-metric
      " does not produce a numeric value.")
    clear-client-plot-data
  ]
end

;; clears the intersection's plot data and the client plot if
;; current-intersection-for-metric is the intersection
to clear-client-plot-data  ;; intersection procedure
  set my-plot-metric ""
  set my-plot-data []
  hubnet-send user-id "Current Metric Value" ""
  if current-intersection-for-client-metric = self
  [
    set-current-plot "Client Plot"
    clear-plot
    set current-intersection-for-client-metric nobody
    set old-intersection-for-client-metric nobody
    set current-client-metric ""
  ]
end

;; plot the data from this pass thru the go procedure
to do-plotting
  if current-intersection-for-client-metric != nobody
  [ update-client-plot ]
  ifelse display-which-metric = old-display-which-metric
  [
    ;; don't plot if no plots should be plotted
    if display-which-metric != 0
    [
      ;; we only need to plot 1 value since the current plot is the same as the plot we are supposed to plot to now
      ifelse display-which-metric = 1
      [ plot-new-value "Stopped Cars" num-cars-stopped ]
      [
        ifelse display-which-metric = 2
        [ plot-new-value "Average Speed of Cars" mean [speed] of turtles ]
        [
          ifelse display-which-metric = 3
          [ plot-new-value "Average Wait Time of Cars" mean [wait-time] of turtles ]
          [
            ;; therefore display-which-metric = 4
            plot-new-value "Stopped Cars" num-cars-stopped
            plot-new-value "Average Wait Time of Cars" mean [wait-time] of turtles
            plot-new-value "Average Speed of Cars" mean [speed] of turtles
          ]
        ]
      ]
    ]
  ]
  [
    ;; otherwise, we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = 0
    [ clear-all-non-client-plot ]
    [
      ifelse display-which-metric = 1
      [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
      [
        ifelse display-which-metric = 2
        [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
        [
          ifelse display-which-metric = 3
          [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
          [
            ;; therefore display-which-metric = 4
            ifelse old-display-which-metric = 1
            [ plot-value-and-lists "Stopped Cars" num-cars-stopped "Average Speed of Cars" speed-data "Average Wait Time of Cars" wait-data ]
            [
              ifelse old-display-which-metric = 2
              [ plot-value-and-lists "Average Speed of Cars" (mean [speed] of turtles) "Stopped Cars" stopped-data "Average Wait Time of Cars" wait-data ]
              [
                ifelse old-display-which-metric = 3
                [ plot-value-and-lists "Average Wait Time of Cars" (mean [wait-time] of turtles) "Stopped Cars" stopped-data "Average Speed of Cars" speed-data ]
                [
                  ;; therefore old-display-which-metric = 0
                  plot-new-list "Stopped Cars" stopped-data
                  plot-new-list "Average Speed of Cars" speed-data
                  plot-new-list "Average Wait Time of Cars" wait-data
                ]
              ]
            ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
end

;; update the client plot with the appropriate information
to update-client-plot
  let current-item []
  set-current-plot "Client Plot"
  ifelse current-intersection-for-client-metric != old-intersection-for-client-metric
  [
    clear-plot
    ;; update client plot with the new intersection's data
    ask current-intersection-for-client-metric
    [
      set current-client-metric my-plot-metric
      let index 0
      repeat length my-plot-data
      [
        set current-item item index my-plot-data
        plotxy first current-item last current-item
        set index index + 1
      ]
    ]
    set old-intersection-for-client-metric current-intersection-for-client-metric
  ]
  [
    ask current-intersection-for-client-metric
    [
      if length my-plot-data != 0
      [ plotxy first last my-plot-data last last my-plot-data ]
    ]
  ]
end

;; clear all the plots that are not the client plot
to clear-all-non-client-plot
  set-current-plot "Stopped Cars"
  clear-plot
  set-current-plot "Average Speed of Cars"
  clear-plot
  set-current-plot "Average Wait Time of Cars"
  clear-plot
end

to plot-new-value [name-of-plot value]
  set-current-plot name-of-plot
  plot value
end

to clear-plots-and-plot-in-new-plot [name-of-plot list-to-plot]
  clear-all-non-client-plot
  plot-new-list name-of-plot list-to-plot
end

to plot-new-list [name-of-plot list-to-plot]
  let index 0

  set-current-plot name-of-plot
  clear-plot
  repeat length list-to-plot
  [
    plot item index list-to-plot
    set index index + 1
  ]
end

to plot-value-and-lists [value-plot value list-plot1 list-to-plot1 list-plot2 list-to-plot2]
  plot-new-value value-plot value
  plot-new-list list-plot1 list-to-plot1
  plot-new-list list-plot2 list-to-plot2
end

;; increases the clock by 1 and cycles phase to the next appropriate value
to clock-tick
  tick
  ;; The phase cycles from 0 to ticks-per-cycle, then starts over.
  set phase phase + 1
  if phase mod ticks-per-cycle = 0
  [ set phase 0 ]
end

;; update the plots to show the current value of display-which-metric
;; done only by the Refresh Plots button
to show-current-metric-in-plots
  if display-which-metric != old-display-which-metric
  [
    ;; we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = 0
    [ clear-all-non-client-plot ]
    [
      ifelse display-which-metric = 1
      [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
      [
        ifelse display-which-metric = 2
        [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
        [
          ifelse display-which-metric = 3
          [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
          [
            ;; therefore display-which-metric = 4
            ifelse old-display-which-metric = 1
            [
              plot-new-list "Average Speed of Cars" speed-data
              plot-new-list "Average Wait Time of Cars" wait-data
            ]
            [
              ifelse old-display-which-metric = 2
              [
                plot-new-list "Stopped Cars" stopped-data
                plot-new-list "Average Wait Time of Cars" wait-data
              ]
              [
                ifelse old-display-which-metric = 3
                [
                  plot-new-list "Stopped Cars" stopped-data
                  plot-new-list "Average Speed of Cars" speed-data
                ]
                [
                  ;; therefore old-display-which-metric = 0
                  plot-new-list "Stopped Cars" stopped-data
                  plot-new-list "Average Speed of Cars" speed-data
                  plot-new-list "Average Wait Time of Cars" wait-data
                ]
              ]
            ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
  if current-intersection-for-client-metric != old-intersection-for-client-metric
  [ update-client-plot ]
end

;; set the current client plot's variable information
to set-current-client-metric-intersection [ intersection-to-plot ]
  set current-intersection-for-client-metric intersection-to-plot
  set current-client-metric [my-plot-metric] of intersection-to-plot
end

;; select an intersection to plot its my-plot-metric in the client plot
to select-intersection-for-client-plot
  if mouse-down?
  [
    let current-patch patch mouse-xcor mouse-ycor
    if ([intersection?] of current-patch) and (-1 != [user-id] of current-patch)
    [
      set-current-client-metric-intersection patch mouse-xcor mouse-ycor
      stop
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "Optional: Zoom In (see Tools in the Menu Bar)"
    "Change the traffic grid (using the sliders GRID-SIZE-X and..."
      "GRID-SIZE-Y) to make enough lights for everyone."
    "Change any other of the settings that you would like to change."
    "For example, if you plan on running Gridlock in..."
      "the MANUAL mode, be sure to have AUTO? set to OFF."
    "Press the SETUP button."
    "Press the LOGIN button."
    "Everyone: Open the HubNet Client on your machine and..."
      "input the IP Address of this computer, type your user name in..."
        "the user name box and press ENTER"
    "Teacher: Once everyone is logged in and has a light..."
      "stop the LOGIN button by pressing it again."
    "Everyone: Whichever mode AUTO? is set for in NetLogo,..."
      "you will control your intersection in a different way."
    "If you have chosen MANUAL,..."
      "you can change the state of your light by pressing..."
        "the CHANGE LIGHT button."
    "If you have chosen AUTO,..."
      "you can change the phase of your light by moving..."
        "the PHASE slider to a different position."
    "If you wish to keep track of your own metric,..."
      "input it in the metric-code input box."
    "Teacher: Once everyone is ready,..."
      "start the simulation by pressing the GO button."
    "Teacher: You may want to view some of the plots."
      "Do this by changing the DISPLAY-WHICH-METRIC slider,..."
        "which changes the plot displayed for everyone."
    "Choose 0 to turn off all the plots..."
      "Choose 1 to see the STOPPED CARS plot..."
        "Choose 2 for the AVERAGE SPEED plot..."
          "Choose 3 for the AVERAGE WAIT plot..."
            "or Choose 4 for all the plots..."

    "Teacher: To run the activity again with the same group,..."
      "stop the model by pressing the GO button, if it is on."
        "Change any of the settings that you would like."
          "Press the SETUP button."
    "Teacher: Once everyone is ready,..."
      "restart the simulation by pressing the GO button."

    "Teacher: To start the simulation over with a new group,..."
      "stop the model by pressing the GO button, if it is on..."
      "Press the RESET button in the Control Center..."
        "and follow these instructions again from the beginning."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; when a command is sent, find out which client sent it and then execute the command
to listen-clients
  while [hubnet-message-waiting?]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [
      give-intersection-coords
      wait 1  ;; we want to give some time for other clients to log in on this round
    ]
    [
      ifelse hubnet-exit-message?
        [ abandon-intersection ]
      [
        ifelse hubnet-message-tag = "Change Light"
          [ manual hubnet-message-source ]
        [
          ifelse hubnet-message-tag = "Phase"
            [ auto hubnet-message-source ]
          [
            if hubnet-message-tag = "metric-code"
              [ new-client-metric hubnet-message-source hubnet-message ]
          ]
        ]
      ]
    ]
  ]
end

;; when a new client logs in, if there are free intersections,
;; assign one of them to that client
;; if this current-id already has an intersection, give the client that intersection.
to give-intersection-coords
  let current-id hubnet-message-source
  ifelse not any? intersections with [user-id = current-id]
  [
      ;; the case where they tried logging in previously but there was no room for them
      ;; or they haven't logged in before
      get-free-intersection current-id
  ]
  [
    ;; otherwise, we already have an intersection for the current-id.
    ;; all we need to do is send where the light is located at
    ask intersections with [user-id = current-id]
    [ hubnet-send current-id "Located At:" (word "(" my-column "," my-row ")") ]
  ]
end

;; when a client disconnects, free up its intersection
to abandon-intersection
  ask intersections with [user-id = hubnet-message-source]
  [
    set user-id -1
    set my-phase 0
    ask patch-at -1 1 [ set plabel "" ]
    set my-plot-data []
    set my-plot-metric ""
    if current-intersection-for-client-metric = self
    [
      set current-client-metric ""
      set current-intersection-for-client-metric nobody
    ]
  ]
end

;; if there are any free intersections, pick one of them at random and give it to the current-id.
;; if there are not any free intersections, toss an error and put error values into the list
to get-free-intersection [current-id]
  ifelse any? intersections with [user-id = -1]
  [
    ;; pick a random intersection that hasn't been taken yet
    ask one-of intersections with [user-id = -1]
    [
      set user-id current-id
      ask patch-at -1 1
      [
        set plabel-color black
        set plabel current-id
      ]
      hubnet-send current-id "Located At:" (word "(" my-column "," my-row ")")
    ]
  ]
  [
    hubnet-send current-id "Located At:" "Not enough lights"
    user-message word "Not enough lights for student with id: " current-id
  ]
end

;; switch the traffic lights at the intersection for the client with user-id
to manual [current-id]
  if not auto?
  [
    ask intersections with [user-id = current-id]
    [
      set green-light-up? (not green-light-up?)
      set-signal-colors
    ]
  ]
end

;; change the value of the phase for the intersection at (xc,yc) to
;; the value passed by the client
to auto [current-id]
  ask intersections with [user-id = current-id]
  [
    set my-phase hubnet-message
  ]
end

;; change the metric to plot on the client plot for the client
;; if it is valid code to generate values to plot
to new-client-metric [ current-id new-metric ]
  ask intersections with [ user-id = current-id ]
  [
    let err __check-syntax (word "plotxy ticks " new-metric)
    ;; don't send an error message if the client sent the empty string
    if new-metric != ""
    [
      hubnet-send user-id "metric-error-message" err
      hubnet-send user-id "Current Metric Value" ""
    ]
    ifelse err = ""
    [
      hubnet-send user-id "Current Metric Value" ""
      set my-plot-metric new-metric
      set my-plot-data []
      if current-intersection-for-client-metric = self
      [
        set-current-plot "Client Plot"
        clear-plot
        set current-client-metric my-plot-metric
      ]
    ]
      [ clear-client-plot-data ]
  ]
end

;; report the user-id of current-intersection-for-metric or the empty string if it is nobody
to-report user-id-of-intersection-for-current-metric
  ifelse current-intersection-for-client-metric = nobody or not is-patch? current-intersection-for-client-metric
  [ report "" ]
  [ report [user-id] of current-intersection-for-client-metric ]
end


; Copyright 2002 Uri Wilensky and Walter Stroup.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
566
96
944
475
-1
-1
10.0
1
10
1
1
1
0
1
1
1
-18
18
-18
18
1
1
0
ticks
30.0

TEXTBOX
4
400
279
463
This slider determines which plot is drawn.\n0=no plots, 1=stopped cars\n2=average speed, 3=average wait\n4=all three plots
11
0.0
0

SLIDER
1
464
166
497
display-which-metric
display-which-metric
0
4
4.0
1
1
NIL
HORIZONTAL

BUTTON
101
209
186
242
login
listen-clients
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
565
500
844
689
Average Wait Time of Cars
Time
Average Wait
0.0
100.0
0.0
5.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" ""

PLOT
283
500
562
689
Average Speed of Cars
Time
Average Speed
0.0
100.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" ""

BUTTON
571
60
689
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
857
60
941
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
780
60
858
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
571
10
941
55
Quick Start Instructions- More in Info Window
quick-start
0
1
11

SLIDER
98
10
195
43
grid-size-y
grid-size-y
1
9
5.0
1
1
NIL
HORIZONTAL

SLIDER
2
10
96
43
grid-size-x
grid-size-x
1
9
5.0
1
1
NIL
HORIZONTAL

SWITCH
99
82
189
115
auto?
auto?
1
1
-1000

SWITCH
191
82
281
115
crash?
crash?
1
1
-1000

SWITCH
2
82
97
115
power?
power?
0
1
-1000

SLIDER
2
46
281
79
number
number
0
400
200.0
1
1
cars
HORIZONTAL

PLOT
1
500
280
689
Stopped Cars
Time
Stopped Cars
0.0
100.0
0.0
100.0
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" ""

BUTTON
187
209
272
242
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
2
118
154
151
simulation-speed
simulation-speed
0
10
7.5
0.1
1
NIL
HORIZONTAL

SLIDER
156
118
281
151
speed-limit
speed-limit
0.1
1
1.0
0.1
1
NIL
HORIZONTAL

BUTTON
13
209
99
242
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
176
154
281
199
Current Phase
phase
3
1
11

SLIDER
2
170
174
203
ticks-per-cycle
ticks-per-cycle
1
100
20.0
1
1
NIL
HORIZONTAL

BUTTON
168
464
280
497
Refresh Plots
show-current-metric-in-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
284
299
562
498
Client Plot
time
NIL
0.0
100.0
0.0
1.0
true
false
"" ""
PENS
"client metric" 1.0 0 -16777216 true "" ""

MONITOR
2
302
242
347
Client Plot's Current Intersection's User Id
user-id-of-intersection-for-current-metric
3
1
11

MONITOR
2
249
562
294
Current Client Metric
current-client-metric
3
1
11

BUTTON
2
353
282
386
Select Intersection To Plot Metric in Client Plot
select-intersection-for-client-plot
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

Students control traffic lights in a real-time traffic simulation.  The teacher controls overall variables, such as the speed limit and the number of cars.  This allows students to explore traffic dynamics, which can lead into many areas of study, from calculus to social studies. This alternate version of the Gridlock HubNet model allows the client to send NetLogo code to the model to be run and produce values which can then be plotted in the a client plot.

Challenge the students to develop strategies to improve traffic and discuss the different ways to measure the quality of traffic.

The coordinates for the traffic lights are based on the first quadrant of the Cartesian plane.  Therefore, the traffic light with the coordinates (0,0) is in the lowest row and the left-most column.  The traffic light above it has coordinates (0,1) and the traffic light to the right of it has (1,0).

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/rp/ps/index.shtml.

Notice: Gridlock Alternate uses features in NetLogo that are considered experimental.  The model may stop running if a student enters a metric in the client that is incorrect, such as dividing by 0, and causes a NetLogo error.  Further, we have not yet tested this alternate version of the model in real classrooms. If you use it in a classroom, we'd like to hear how it goes; contact us at feedback@ccl.northwestern.edu.

## HOW TO USE IT

### Quickstart Instructions:

Contains instructions as to how to quickly setup the model, and clients to run this activity.  The instructions can be found below and can be seen progressively in the Quick Start instructions monitor in the Interface:

Teacher: Follow these directions to setup the HubNet activity.
Optional: Zoom In (see Tools in the Menu Bar)
Change the traffic grid (using the sliders GRID-SIZE-X and GRID-SIZE-Y) to make enough lights for everyone.
Change any other of the settings that you would like to change.  For example, if you plan on running Gridlock in the MANUAL mode, be sure to have AUTO? set to OFF.

Press the SETUP button.
Press the LOGIN button.

Everyone: Open up a HubNet Client on your machine and input the IP Address of this computer, type your user name in the user name box and press ENTER.
Teacher: Once everyone is logged in and has a light stop the LOGIN button by pressing it again.

Everyone: Whichever mode AUTO? is set for in NetLogo, you will control your intersection in a different way:
If you have chosen MANUAL, you can change the state of your light by pressing the CHANGE LIGHT button.
If you have chosen AUTO, you can change the phase of your light by moving the PHASE slider to a different position.
If you wish to keep track of your own metric, input it in the metric-code input box.

Teacher: Once everyone is ready, start the simulation by pressing the GO button.

Teacher: You may want to view some of the plots.  Do this by changing the DISPLAY-WHICH-METRIC slider, which changes the plot displayed for everyone.
- Choose 0 to turn off all the plots.
- Choose 1 to see the STOPPED CARS plot.
- Choose 2 for the AVERAGE SPEED OF CARS plot.
- Choose 3 for the AVERAGE WAIT TIME OF CARS plot.
- Choose 4 for all the plots.

Teacher: To run the activity again with the same group, stop the model by pressing the GO button if it is on.  Change the values of the sliders and switches to the values you want for the new run.  Press the SETUP button.  Once everyone is ready, restart the simulation by pressing the GO button.

Teacher: To start the simulation over with a new group, stop the model by pressing the GO button if it is on, press the RESET button in the Control Center and follow these instructions again from the beginning.

### Buttons

SETUP - generates a new traffic grid based on the current GRID-SIZE-X and GRID-SIZE-Y and NUM-CARS number of cars.  This also clears all the plots.  If the size of the grid has changed the clients will be assigned to new intersections.
GO - runs the simulation indefinitely
LOGIN - allows users to log into the activity without running the model or collecting data
REFRESH PLOTS - redraws the plots based on the current value of DISPLAY-WHICH-METRIC.  Useful for looking at different plots when GO is off.
SELECT INTERSECTION TO PLOT METRIC IN CLIENT PLOT - allows you to use the mouse to click on an intersection associated with a HubNet client to plot that client's submitted metric in the CLIENT PLOT.

### Sliders

SPEED-LIMIT - sets the maximum speed for the cars
NUMBER - the number of cars in the simulation (you must press the SETUP or RE-RUN buttons to see the change)
SIMULATION-SPEED - the speed at which the simulation runs
TICKS-PER-CYCLE - sets the maximum value that the phase can be.  This has no effect when the model is run with AUTO? false.  Also, the phase that each user chooses is scaled to be less than or equal to this value.
GRID-SIZE-X - sets the number of vertical roads there are (you must press the SETUP button to see the change)
GRID-SIZE-Y - sets the number of horizontal roads there are (you must press the SETUP button to see the change)
DISPLAY-WHICH-METRIC - determines which plot is drawn in NetLogo:
- 0=No Plots.
- 1=STOPPED CARS
- 2=AVERAGE SPEED OF CARS
- 3=AVERAGE WAIT TIME OF CARS
- 4=All three plots.

### Switches

CRASH? - toggles car crashing
POWER? - toggles the presence of traffic lights
AUTO? - toggles between automatic mode, where the students' lights change on a cycle, and manual in which students directly control the lights with their clients. Lights which aren't associated with clients always change on a cycle.

### Plots

STOPPED CARS - displays the number of stopped cars over time
AVERAGE SPEED OF CARS - displays the average speed of cars over time
AVERAGE WAIT TIME OF CARS - displays the average time cars are stopped over time
CLIENT PLOT - displays the currently selected client's submitted metric over time.  You select a client's intersection by using the SELECT INTERSECTION TO PLOT METRIC IN CLIENT PLOT button.

### Client Information

After logging in, the client interface will appear for the students.  The controls for manual and automatic mode are both included, but which one works is based on the setting of the AUTO? switch in NetLogo.  In MANUAL mode, click the CHANGE LIGHT button to switch the state of the light you control.  In AUTO mode, move the PHASE slider to change the phase for your light.  The phase determines what percent of the way through the cycle to switch on.

If the users wish to send a metric of their own making that measures the efficiency of the traffic to the model, they can do so with the text input interface element called METRIC-CODE.  The user should input a valid NetLogo expression that evaluates to a number.  Then if this client is selected during a run, the values of the metric will be plotted in the CLIENT PLOT.  If you don't type a valid NetLogo expression that evaluates to a number, an error message will appear in the METRIC-ERROR-MESSAGE monitor.

The clients also contain the same four plots as NetLogo and will show exactly the same data as NetLogo if plot mirroring is enabled.

## THINGS TO NOTICE

When cars have stopped at a traffic light, and then they start moving again, the traffic jam will move backwards even though the cars are moving forwards.  Why is this?  Discuss in your class possible reasons for this phenomena.

## THINGS TO TRY

Try changing the speed limit for the cars.  How does this affect the overall efficiency of the traffic flow?  Are fewer cars stopping for a shorter amount of time?  Is the average speed of the cars higher or lower than before?

Try changing the number of cars on the roads.  Does this affect the efficiency of the traffic flow?

How about changing the speed of the simulation?  Does this affect the efficiency of the traffic flow?

Using HubNet, try running this simulation with AUTO? being true and AUTO? being false.  Is it harder to make the traffic move well using one scheme or the other?  Why?

Using HubNet, try running this simulation with AUTO? being true.  Try to find a way of setting the phases of the traffic lights so that the average speed of the cars is the highest.  Now try to minimize the number of stopped cars.  Now try to decrease the average wait time of the cars.  Is there any correlation between these different metrics?

Try coming up with new and good ways of measuring the efficiency of the traffic flow.

## EXTENDING THE MODEL

Currently, the maximum speed limit (found in the SPEED-LIMIT slider) for the cars is 1.0.  This is due to the fact that the cars must look ahead the speed that they are traveling to see if there are cars ahead of them.  If there aren't, they speed up.  If there are, they slow down.  Looking ahead for a value greater than 1 is a little bit tricky.  Try implementing the correct behavior for speeds greater than 1.

## NETLOGO FEATURES

This model uses a text input interface element in the clients to allow clients to send text to the model.

It also uses the experimental `__check-syntax` primitive to try to ensure a string is safe to pass to the run-result primitive.

Together, these features allow the client to send NetLogo code to the model to be run and produce values which can then be plotted in the CLIENT PLOT.

If the code that is executed is valid NetLogo syntax and doesn't produce any bad side-effects, such as clearing the View or having turtles die, everything should work fine.  However, there is no easy way of ensuring this currently.  As a result, code which may be unsafe to the model could be executed and may cause the model to stop unexpectedly or give a runtime error when trying to plot to the CLIENT PLOT.  We will be improving these features in future versions of NetLogo.

## RELATED MODELS

- "Traffic Basic": a simple model of the movement of cars on a highway.

- "Traffic Basic Utility": a version of "Traffic Basic" including a utility function for the cars.

- "Traffic Basic Adaptive": a version of "Traffic Basic" where cars adapt their acceleration to try and maintain a smooth flow of traffic.

- "Traffic Basic Adaptive Individuals": a version of "Traffic Basic Adaptive" where each car adapts individually, instead of all cars adapting in unison.

- "Traffic 2 Lanes": a more sophisticated two-lane version of the "Traffic Basic" model.

- "Traffic Intersection": a model of cars traveling through a single intersection.

- "Traffic Grid": a model of traffic moving in a city grid, with stoplights at the intersections.

- "Traffic Grid Goal": a version of "Traffic Grid" where the cars have goals, namely to drive to and from work.

- "Gridlock HubNet": a version of "Traffic Grid" where students control traffic lights in real-time.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. and Stroup, W. (2002).  NetLogo Gridlock Alternate HubNet model.  http://ccl.northwestern.edu/netlogo/models/GridlockAlternateHubNet.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the HubNet software as:

* Wilensky, U. & Stroup, W. (1999). HubNet. http://ccl.northwestern.edu/netlogo/hubnet.html. Center for Connected Learning and Computer-Based Modeling, Northwestern University. Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2002 Uri Wilensky and Walter Stroup.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

This activity and associated models and materials were created as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

<!-- 2002 Stroup -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
need-to-manually-make-preview-for-this-model
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
492
10
862
380
0
0
0
1
1
1
1
1
0
1
1
1
-18
18
-18
18

BUTTON
192
68
316
101
Change Light
NIL
NIL
1
T
OBSERVER
NIL
NIL

SLIDER
192
101
316
134
Phase
Phase
0.0
99.0
0
1.0
1
NIL
HORIZONTAL

TEXTBOX
13
10
184
175
Affect the state of your light by pressing the Change Light button, or changing the value of the Phase slider. The Phase slider controls the point in the cycle at which your light with change, and represents a percentage of the total cycle time.
11
0.0
0

MONITOR
203
10
316
59
Located At:
NIL
3
1

MONITOR
7
377
488
426
metric-error-message
NIL
3
1

INPUTBOX
7
136
487
374
metric-code
null
1
1
String (reporter)

MONITOR
318
85
487
134
Current Metric Value
NIL
3
1

@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
